'use client'

import Link from 'next/link'
import { useState, useEffect } from 'react'
import CartIcon from './components/CartIcon'

// Convert EUR to BGN and round to nearest 0.10
const convertToBGN = (eur: number | null | undefined): string => {
  if (!eur || eur === 0) return '0.00'
  const bgn = eur * 1.95583
  // Round to nearest 0.10 (e.g., 58.68 -> 58.70, 58.86 -> 58.90)
  const rounded = Math.round(bgn * 10) / 10
  return rounded.toFixed(2)
}

interface Product {
  id: number
  name: string
  price: number
  images: string[]
  category: string
  discount?: number
}

const categories = [
  { id: 'clothing', name: 'Дрехи', icon: '👕' },
  { id: 'dresses', name: 'Рокли', icon: '👗' },
  { id: 'underwear', name: 'Бельо', icon: '👙' },
  { id: 'swimwear', name: 'Бански', icon: '🏊' },
  { id: 'sportswear', name: 'Клинове/Спорт', icon: '🧘' },
  { id: 'accessories', name: 'Аксесоари', icon: '👜' },
]

export default function Home() {
  const [selectedCategory, setSelectedCategory] = useState<string>('clothing')
  const [products, setProducts] = useState<Product[]>([])
  const [loading, setLoading] = useState(true)
  const [showSuccessMessage, setShowSuccessMessage] = useState(false)

  useEffect(() => {
    // Check for order success parameter
    if (typeof window !== 'undefined') {
      const params = new URLSearchParams(window.location.search)
      if (params.get('order') === 'success') {
        setShowSuccessMessage(true)
        // Remove the parameter from URL
        window.history.replaceState({}, '', window.location.pathname)
        // Hide message after 5 seconds
        setTimeout(() => setShowSuccessMessage(false), 5000)
      }
    }

    fetch('/api/products')
      .then(res => res.json())
      .then(data => {
        setProducts(data)
        setLoading(false)
      })
      .catch(err => {
        console.error('Error loading products:', err)
        setLoading(false)
      })
  }, [])

  const filteredProducts = selectedCategory
    ? products.filter(p => p.category === selectedCategory)
    : products

  return (
    <div className="min-h-screen bg-[#F5F5F5]">
      {/* Success Message */}
      {showSuccessMessage && (
        <div className="bg-green-50 border-l-4 border-green-500 text-green-700 p-4 mb-4 mx-4 mt-4 rounded-lg shadow-md max-w-7xl mx-auto">
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <svg className="h-5 w-5 text-green-500 mr-2" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
              </svg>
              <p className="font-medium">Поръчката е изпратена успешно! Ще се свържем с вас скоро.</p>
            </div>
            <button
              onClick={() => setShowSuccessMessage(false)}
              className="text-green-700 hover:text-green-900 ml-4"
            >
              <svg className="h-5 w-5" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clipRule="evenodd" />
              </svg>
            </button>
          </div>
        </div>
      )}

      {/* Announcement Bar */}
      <div className="bg-black text-white text-center py-2 text-sm font-medium">
        БЕЗПЛАТНА ДОСТАВКА НАД 35€
      </div>

      {/* Header */}
      <header className="bg-white">
        <div className="max-w-7xl mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            {/* Profile Picture */}
            <div className="w-10 h-10 rounded-full bg-light-pink flex items-center justify-center overflow-hidden">
              <svg
                xmlns="http://www.w3.org/2000/svg"
                className="h-6 w-6 text-dark-grey"
                fill="none"
                viewBox="0 0 24 24"
                stroke="currentColor"
                strokeWidth={2}
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"
                />
              </svg>
            </div>
            {/* Domain Name */}
            <h1 className="text-xl md:text-2xl font-bold text-dark-grey">seksi.bg</h1>
            {/* Cart Icon */}
            <CartIcon />
          </div>
        </div>
      </header>

      {/* Categories */}
      <section className="bg-white py-4">
        <div className="max-w-7xl mx-auto px-4">
          <div className="flex flex-wrap justify-center gap-3">
            {categories.map((category) => (
              <button
                key={category.id}
                onClick={() => setSelectedCategory(category.id)}
                className={`px-4 py-2 rounded-full transition-all text-sm md:text-base ${
                  selectedCategory === category.id
                    ? 'bg-light-pink text-dark-grey font-medium'
                    : 'text-medium-grey hover:bg-gray-100'
                }`}
              >
                {category.name}
              </button>
            ))}
          </div>
        </div>
      </section>

      {/* Products Grid */}
      <section className="max-w-7xl mx-auto px-4 py-6">
        {loading ? (
          <div className="text-center py-12">
            <p className="text-medium-grey">Зареждане на продукти...</p>
          </div>
        ) : filteredProducts.length === 0 ? (
          <div className="text-center py-12">
            <p className="text-medium-grey">Няма продукти в тази категория.</p>
          </div>
        ) : (
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 md:gap-6">
            {filteredProducts.map((product) => (
            <Link
              key={product.id}
              href={`/product/${product.id}`}
              className="group"
            >
              <div className="relative aspect-[3/4] bg-white rounded-lg mb-2 overflow-hidden">
                {product.discount && (
                  <div className="absolute top-2 right-2 bg-light-pink text-dark-grey text-xs font-semibold px-2 py-1 rounded z-10">
                    -{product.discount}%
                  </div>
                )}
                {product.images && product.images.length > 0 ? (
                  <img
                    src={`/images/products/${product.images[0]}`}
                    alt={product.name}
                    className="w-full h-full object-cover"
                    onError={(e) => {
                      const target = e.currentTarget as HTMLImageElement
                      target.style.display = 'none'
                      const placeholder = target.parentElement?.querySelector('.image-placeholder') as HTMLElement
                      if (placeholder) placeholder.style.display = 'flex'
                    }}
                  />
                ) : null}
                <div className={`image-placeholder w-full h-full bg-gradient-to-br from-gray-200 to-gray-300 ${product.images && product.images.length > 0 ? 'hidden' : 'flex'} items-center justify-center`}>
                  <span className="text-medium-grey text-xs">Изображение</span>
                </div>
              </div>
              <h3 className="text-sm font-medium text-dark-grey mb-1 group-hover:text-medium-grey transition">
                {product.name}
              </h3>
              <p className="text-base font-semibold text-dark-grey">
                €{product.price ? product.price.toFixed(0) : '0'} ({convertToBGN(product.price)} лв)
              </p>
            </Link>
            ))}
          </div>
        )}

      </section>

      {/* Footer */}
      <footer className="bg-white border-t border-gray-200 mt-12">
        <div className="max-w-7xl mx-auto px-4 py-8">
          <div className="grid grid-cols-2 md:grid-cols-4 gap-6 mb-6">
            <div>
              <h3 className="text-sm font-semibold text-dark-grey mb-3">За нас</h3>
              <ul className="space-y-2">
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    За нас
                  </a>
                </li>
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Контакти
                  </a>
                </li>
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Доставка и връщане
                  </a>
                </li>
              </ul>
            </div>
            <div>
              <h3 className="text-sm font-semibold text-dark-grey mb-3">Помощ</h3>
              <ul className="space-y-2">
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Често задавани въпроси
                  </a>
                </li>
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Размери
                  </a>
                </li>
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Условия за ползване
                  </a>
                </li>
              </ul>
            </div>
          </div>
          <div className="pt-6 border-t border-gray-200 text-center">
            <p className="text-xs text-medium-grey">
              © {new Date().getFullYear()} seksi.bg. Всички права запазени.
            </p>
          </div>
        </div>
      </footer>
    </div>
  )
}
